/*************************************************************
  techday23.cpp - Line following robot based ESP8288 with PID

  !! Disconnect RX pin during flashing !!

  Program is connected to Blynk to be controlled from IOS or Android
  (c) 2023 Alfred Velthuis and Gert Maneschijn
 *************************************************************/
// #define BLYNK_DEBUG // For debugging Blynk

// #define DEBUG // AV-For debugging inmternals Techday23 routines

/* Fill-in information from Blynk Device Info here */
#define BLYNK_TEMPLATE_ID "TMPL8tAN0rfA"
#define BLYNK_TEMPLATE_NAME "TechDay23"

//#define ALFRED


// Enter the right Token.
#define BLYNK_AUTH_TOKEN "jfUSuy1gN1E20QSIBQOznCas649zOJua"

/* Comment this out to disable prints and save space */
//#define BLYNK_PRINT Serial

// Includes
#include <LittleFS.h>
#include <ESP8266WiFi.h>
#include <BlynkSimpleEsp8266.h>
#include <PID_v1.h>

double dSetpointSteer, dInputSteer, dOutputSteer;
double dSetpointSpeed, dInputSpeed, dOutputSpeed;
int iMinSpeed = 170;      // minimal speed for motors to start
int iMinMinSpeed = 155;   // minimal speed for motor before stopping
int iSpeedOffset = 60;    // Speed offset to enable steering with JoyStick
int iPidSampleTime = 100; // default 100ms sample time for PID

unsigned long ulPreviousMillis = 0;

// Specify the links and initial tuning parameters
double dKpSteer = 25, dKiSteer = 0, dKdSteer = 0;
double dKpSpeed = 1, dKiSpeed = 0, dKdSpeed = 0;

PID pidSteer(&dInputSteer, &dOutputSteer, &dSetpointSteer, dKpSteer, dKiSteer, dKdSteer, DIRECT);
PID pidSpeed(&dInputSpeed, &dOutputSpeed, &dSetpointSpeed, dKpSpeed, dKiSpeed, dKdSpeed, DIRECT);

// Your WiFi credentials.
// Set password to "" for open networks.
//
// Defaults (configured as Hotspot so new Wifi credentials can be entered through Blynk)
// char ssid[] = "rdwtechday23";
// char pass[] = "!Nodemcu23";
#define SSID "rdwtechday23"
#define PASS "!Nodemcu23"

// Assign MCU pins to ESP motorshield board
// LEFT is motor A ans right is motor B
int LEFTPWM = D2;  // GPIO 5
int LEFTDIR = D3;  // GPIO 0
int RIGHTPWM = D1; // GPIO 4
int RIGHTDIR = D4; // GPIO 2

// Assign Line Tracking Sensors
int S1 = D0; // GPIO 16
int S2 = 3;  // GPIO  3  RXD0  Disconnect during flashing!!!
int S3 = D6; // GPIO 12
int S4 = D7; // GPIO 13
int S5 = D5; // GPIO 14

// Default is manual operation through Joystick Widget
bool bOperatingMode = true;

WidgetTerminal terminal(V8);

BlynkTimer timer;

// Joystick on/off
BLYNK_WRITE(V0)
{
  // Set incoming value from pin V0 to a variable
  int value = param.asInt();

  if (value == 0)
  {
    bOperatingMode = false;
  }
  else
  {
    bOperatingMode = true;
  }
  analogWrite(RIGHTPWM, 0);
  analogWrite(LEFTPWM, 0);
}

// Handling Speed Slider
BLYNK_WRITE(V1)
{
  int value = param.asInt();

  iMinSpeed = value;
}

// Handling dKp Steer Slider
BLYNK_WRITE(V2)
{
  int value = param.asInt();

  dKpSteer = (double)value;
}

// Handling dKi Steer Slider
BLYNK_WRITE(V6)
{
  int value = param.asInt();

  dKiSteer = (double)value;
}

// Handling dKd Steer Slider
BLYNK_WRITE(V7)
{
  int value = param.asInt();

  dKdSteer = (double)value;
}

// Change PID sample time
BLYNK_WRITE(V10)
{
  int value = param.asInt();

  iPidSampleTime = value;
  pidSteer.SetSampleTime(iPidSampleTime);
}

// Change Speed Offset
BLYNK_WRITE(V11)
{
  int value = param.asInt();

  iSpeedOffset = value;
}

BLYNK_WRITE(V12)
{
  File cfg;

  Serial.println(param.asStr());

  cfg = LittleFS.open("/techday23.cfg", "w");
  cfg.printf("%s\n", param.asStr());
  cfg.close();
}

// Handling Joystick data
BLYNK_WRITE(V4)
{
  int iProportionalSpeed = 0;

  int x = param[0].asInt();
  int y = param[1].asInt();

  // If Joystick off get out
  if (!bOperatingMode)
  {
    return;
  }

  // determine speed according Joystick
  iProportionalSpeed = abs(map(y, 0, 255, -128, 128)) * 2;

  // determine direction according Joystick
  x = map(x, 0, 255, -1, 1);
  y = map(y, 0, 255, -1, 1);

#ifdef DEBUG
  Serial.printf("[%05lu]-X:%03d--Y:%03d--Speed:%03d\n", millis(), x, y, iProportionalSpeed);
#endif

  // If not action in Joystick fall back to rest (No Speed, No PWM)
  if (x == 0 && y == 0)
  {
    analogWrite(RIGHTPWM, 0);
    analogWrite(LEFTPWM, 0);
    iMinSpeed = 0;
    return;
  }
  else
  {
    iMinSpeed = min(iProportionalSpeed, 255);
  }

  if (x == -1 && y == -1)
  { // Backward and Left
    analogWrite(RIGHTPWM, iMinSpeed);
    digitalWrite(RIGHTDIR, HIGH);

    analogWrite(LEFTPWM, iMinSpeed - iSpeedOffset);
    digitalWrite(LEFTDIR, HIGH);
  }
  else if (x == -1 && y == 0)
  { // Left Turn
    iMinSpeed = max(iMinSpeed, 180);
    analogWrite(RIGHTPWM, iMinSpeed);
    digitalWrite(RIGHTDIR, LOW);

    analogWrite(LEFTPWM, iMinSpeed);
    digitalWrite(LEFTDIR, HIGH);
  }
  else if (x == -1 && y == 1)
  { // Forward and Left
    analogWrite(RIGHTPWM, iMinSpeed);
    digitalWrite(RIGHTDIR, LOW);

    analogWrite(LEFTPWM, iMinSpeed - iSpeedOffset);
    digitalWrite(LEFTDIR, LOW);
  }
  else if (x == 0 && y == -1)
  { // Backward
    analogWrite(RIGHTPWM, iMinSpeed);
    digitalWrite(RIGHTDIR, HIGH);

    analogWrite(LEFTPWM, iMinSpeed);
    digitalWrite(LEFTDIR, HIGH);
  }
  else if (x == 0 && y == 0)
  { // Stay
    analogWrite(RIGHTPWM, LOW);
    digitalWrite(RIGHTDIR, LOW);

    analogWrite(LEFTPWM, LOW);
    digitalWrite(LEFTDIR, LOW);
  }
  else if (x == 0 && y == 1)
  { // Forward
    analogWrite(RIGHTPWM, iMinSpeed);
    digitalWrite(RIGHTDIR, LOW);

    analogWrite(LEFTPWM, iMinSpeed);
    digitalWrite(LEFTDIR, LOW);
  }
  else if (x == 1 && y == -1)
  { // Backward and Right
    analogWrite(RIGHTPWM, iMinSpeed - iSpeedOffset);
    digitalWrite(RIGHTDIR, HIGH);

    analogWrite(LEFTPWM, iMinSpeed);
    digitalWrite(LEFTDIR, HIGH);
  }
  else if (x == 1 && y == 0)
  { // Right turn
    iMinSpeed = max(iMinSpeed, 180);
    analogWrite(RIGHTPWM, iMinSpeed);
    digitalWrite(RIGHTDIR, HIGH);

    analogWrite(LEFTPWM, iMinSpeed);
    digitalWrite(LEFTDIR, LOW);
  }
  else if (x == 1 && y == 1)
  { // Forward and Right
    analogWrite(RIGHTPWM, iMinSpeed - iSpeedOffset);
    digitalWrite(RIGHTDIR, LOW);

    analogWrite(LEFTPWM, iMinSpeed);
    digitalWrite(LEFTDIR, LOW);
  }
}

// This function is called every time the device is connected to the Blynk.Cloud
BLYNK_CONNECTED()
{
  // Change Web Link Button message to "Congratulations!"
  Blynk.setProperty(V3, "offImageUrl", "https://www.velthuis.info/techdays2023-1.png");
  Blynk.setProperty(V3, "onImageUrl", "https://www.velthuis.info/techdays2023-2.png");
  Blynk.setProperty(V3, "url", "https://www.velthuis.info");
}

// This function sends Arduino's uptime every second to Virtual Pin 2.
void myTimerEvent()
{
  char szBuffer[100];

  sprintf(szBuffer, "%d-%d-%d-%d-%d   %05lu %07.3f", digitalRead(S1), digitalRead(S2), digitalRead(S3), digitalRead(S4), digitalRead(S5), millis() / 1000, dOutputSteer);

  Blynk.virtualWrite(V5, szBuffer);
  Blynk.virtualWrite(V1, iMinSpeed);
}

static void waitForWifiToConnect(int maxRetries)
{
  int retryCount = 0;
  while ((WiFi.status() != WL_CONNECTED) && (retryCount < maxRetries))
  {
    delay(500);
    Serial.printf("[%lu]Wifi not connected...\n", millis());
    ++retryCount;
  }
}

void setup()
{
  IPAddress MyIP;
  File configFile;
  char szSSID[30];
  char szPASS[30];
  String strTmp;

  // Debug console
  Serial.begin(115200);
  delay(2000);

  strcpy(szSSID, SSID);
  strcpy(szPASS, PASS);

  if (LittleFS.begin())
  {
    Serial.printf("[%lu]mounted file system...\n", millis());
    if (LittleFS.exists("/techday23.cfg"))
    {
      // file exists, reading and loading
      Serial.printf("[%lu]Reading config file...\n", millis());
      configFile = LittleFS.open("/techday23.cfg", "r");

      while (configFile.available())
      {
        String sLine = configFile.readStringUntil('\n');
        int index = sLine.indexOf(' ');
        strTmp = sLine.substring(0, index);
        strTmp.trim();
        strTmp.toCharArray(szSSID, 30);
        strTmp = sLine.substring(index + 1);
        strTmp.trim();
        strTmp.toCharArray(szPASS, 30);
        Serial.printf("s[%s]\n", szSSID);
        Serial.printf("p[%s]\n", szPASS);
      }
      configFile.close();
    }
  }

  WiFi.mode(WIFI_STA);
  if (szPASS && strlen(szPASS))
  {
    WiFi.begin(szSSID, szPASS);
  }
  else
  {
    WiFi.begin(szSSID);
  }
  waitForWifiToConnect(40);
  if (WiFi.status() != WL_CONNECTED)
  {
    Serial.printf("[%lu]Wifi not connected...\n", millis());
    configFile = LittleFS.open("/techday23.cfg", "w");
    configFile.printf("%s %s\n", SSID, PASS);
    configFile.close();
    Serial.printf("[%lu]Reset...\n", millis());
    ESP.restart();
  }
  // WiFi.disconnect();

  // You can also specify server:
  // Blynk.begin(BLYNK_AUTH_TOKEN, ssid, pass, "blynk.cloud", 80);
  // Blynk.begin(BLYNK_AUTH_TOKEN, ssid, pass, IPAddress(192,168,1,100), 8080);
  Blynk.begin(BLYNK_AUTH_TOKEN, szSSID, szPASS);
  Blynk.virtualWrite(V0, 1);

  terminal.println();
  terminal.println(F("--------------------------------------------------------------"));
  terminal.println(F("RDW TechDay 2023  [Blynk v" BLYNK_VERSION "] Device started"));
  terminal.println(F("--------------------------------------------------------------"));
  terminal.print("Chip ID: ");
  terminal.println(ESP.getChipId());
  terminal.println(WiFi.localIP());
  terminal.println(WiFi.macAddress());
  terminal.flush();

  // Setup a function to be called every 100ms
  timer.setInterval(1000L, myTimerEvent);

  // Initialize GPIOs
  pinMode(RIGHTPWM, OUTPUT);
  pinMode(LEFTPWM, OUTPUT);
  pinMode(RIGHTDIR, OUTPUT);
  pinMode(LEFTDIR, OUTPUT);

  // Line detection sensor
  pinMode(S1, INPUT);
  pinMode(S2, INPUT);
  pinMode(S3, INPUT);
  pinMode(S4, INPUT);
  pinMode(S5, INPUT);

  // Initialize PID parameters
  dSetpointSteer = 10; // no error is target
  dSetpointSpeed = 0;

  // turn the PID on
  pidSteer.SetSampleTime(iPidSampleTime);
  pidSteer.SetMode(AUTOMATIC);
}

void loop()
{
  int A1, A2, A3, A4, A5, iSteerError = 10, i;

  unsigned long ulCurrentMillis;

  Blynk.run();
  timer.run();

  A1 = digitalRead(S1);
  A2 = digitalRead(S2);
  A3 = digitalRead(S3);
  A4 = digitalRead(S4);
  A5 = digitalRead(S5);

  // Joystick COntrol
  if (bOperatingMode)
  {
    return;
  }

  ulCurrentMillis = millis();
  if (ulCurrentMillis - ulPreviousMillis < (unsigned long)iPidSampleTime)
  {
    return;
  }
  else
  {
    ulPreviousMillis = ulCurrentMillis;
  }

  if ((A1 == 1 && A2 == 1 && A3 == 1 && A4 == 1 && A5 == 1) || (A1 == 0 && A2 == 0 && A3 == 0 && A4 == 0 && A5 == 0))
  {
    analogWrite(RIGHTPWM, 0);
    analogWrite(LEFTPWM, 0);
    iSteerError = 10;
  }
  else
  {
    if (A1 == 1 && A2 == 1 && A3 == 0 && A4 == 1 && A5 == 1)
      iSteerError = 10;
    else if (A1 == 1 && A2 == 0 && A3 == 0 && A4 == 1 && A5 == 1)
      iSteerError = 9;
    else if (A1 == 1 && A2 == 0 && A3 == 1 && A4 == 1 && A5 == 1)
      iSteerError = 8;
    else if (A1 == 0 && A2 == 0 && A3 == 1 && A4 == 1 && A5 == 1)
      iSteerError = 7;
    else if (A1 == 0 && A2 == 1 && A3 == 1 && A4 == 1 && A5 == 1)
      iSteerError = 6;
    else if (A1 == 1 && A2 == 1 && A3 == 0 && A4 == 0 && A5 == 1)
      iSteerError = 11;
    else if (A1 == 1 && A2 == 1 && A3 == 1 && A4 == 0 && A5 == 1)
      iSteerError = 12;
    else if (A1 == 1 && A2 == 1 && A3 == 1 && A4 == 0 && A5 == 0)
      iSteerError = 13;
    else if (A1 == 1 && A2 == 1 && A3 == 1 && A4 == 1 && A5 == 0)
      iSteerError = 14;

    // straight ahead
    if (iSteerError == 10)
    {
      analogWrite(RIGHTPWM, iMinSpeed); // rechter wiel
      analogWrite(LEFTPWM, iMinSpeed);  // linker wiel
      Serial.printf("Straight ahead: %03d\n", iMinSpeed);
      return;
    }

    pidSteer.SetTunings(dKpSteer, dKiSteer, dKdSteer);
    pidSteer.SetOutputLimits(-255, 255);
    dInputSteer = (double)iSteerError;
    pidSteer.Compute();

    digitalWrite(RIGHTDIR, LOW); // drive forward right
    digitalWrite(LEFTDIR, LOW);  // drive forward left

    // Forward and Left
    if (dOutputSteer > 0)
    {

      i = max(0, iMinSpeed - (int)abs(dOutputSteer));
      // i = max(iMinMinSpeed, iMinSpeed - (int) abs(dOutputSteer));
      analogWrite(LEFTPWM, i); // linker wiel
      Serial.printf("PWM-L: %03d", i);

      i = min(255, iMinSpeed + (int)abs(dOutputSteer));
      analogWrite(RIGHTPWM, i); // rechter wiel
      Serial.printf(" PWM-R: %03d Steer:%03d--PIDOutput:%+03d\n", i, iSteerError, (int)dOutputSteer);
    }
    // Forward and Right
    if (dOutputSteer < 0)
    {
      i = min(255, iMinSpeed + (int)abs(dOutputSteer));
      analogWrite(LEFTPWM, i); // linker wiel
      Serial.printf("PWM-L: %03d", i);

      i = max(0, iMinSpeed - (int)abs(dOutputSteer));
      // i = max(iMinMinSpeed, iMinSpeed - (int) abs(dOutputSteer));
      analogWrite(RIGHTPWM, i); // rechter wiel
      Serial.printf(" PWM-R: %03d Steer:%03d--PIDOutput:%+03d\n", i, iSteerError, (int)dOutputSteer);
    }
  }
}
